// ============================================================================
// ============================================================================
// ============================================================================
// ==                                                                        ==
// == Name    : TheEmuLib.Emu_Quantise_RGBA.A.fsh                            ==
// == Type    : Fragment shader                                              ==
// == Version : 1.0.2 (2017/04/02)                                           ==
// == Creator : TheEmu © TheEmu 2017, Some Rights Reserved                   ==
// == Licence : Creative Commons Attribution-ShareAlike 4.0                  ==
// ==           http://creativecommons.org/licences/by-sa/4.0                ==
// ==                                                                        ==
// == Purpose: To reduce the variation in the colour components for an image ==
// == by mapping them to small number of levels, i.e. to posterise an image. ==
// ==                                                                        ==
// == Description: Each component, Red, Green, Blue and Alpha, of the source ==
// == image the colour, c, is modified using                                 ==
// ==                                                                        ==
// ==           c' = floor ( c*QN + Q0 ) / QN;                               ==
// ==                                                                        ==
// == where the constants QN and Q0 are the quantisation factor and offset.  ==
// ==                                                                        ==
// == Q0 defaults to 0.0 but other values can be used, in particular a value ==
// == of 0.5 effectively changes the floor (round down) function into a nint ==
// == (round to nearest integer) function and a value of 1.0 changes it into ==
// == a ceil (round up) function. QN is normaly a small positive integer and ==
// == specifies the number of quantisation levels but non integer values can ==
// == be used.                                                               ==
// ==                                                                        ==
// == This file is a member of The Emu's shader library.                     ==
// ==                                                                        ==
// == ====================================================================== ==
// ==                                                                        ==
// == Update history:                                                        ==
// ==                                                                        ==
// ==   2017/04/07 - v1.0.0 - Initial version.                               ==
// ==                                                                        ==
// ============================================================================
// ============================================================================
// ============================================================================

// ============================================================================
// == Standard shader inputs ==================================================
// ============================================================================

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// The image that is to be manipulated.

uniform sampler2D iChannel0;

// ============================================================================
// == Imports from TheEmuLib ==================================================
// ============================================================================
//
// The GLSL shader language currently provides no mechanism for importing  any
// elements that are defined in other modules, not even C's crude source level
// #include mechanism. In the absence of anything better TheEmuLib handles any
// imports by manualy copying relevent utility code snippets from the  sources
// in the Shader Lib.Inc directory. This is very crude but I have attempted to
// be systematic in the way in which this is presented in the library sources.
//
// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Macros from TheEmuLib.Emu_Common_Utilities.lib.src

#define EMU_DEFAULT(type,x,default_value) ( (x==type(0.0)) ? (default_value) : (x) )

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Functions from TheEmuLib.Emu_Coordinate_Normalisation.lib.src

vec2 Emu_Normalise_to_Window ( vec2 xy ) { return xy / u_WindowSize.xy; }

// ============================================================================
// == Shader specific inputs ==================================================
// ============================================================================

// EmuLib standard scale and hotspot parameters.

uniform vec2 Emu_Quantise_RGBA_scale;
uniform vec2 Emu_Quantise_RGBA_hotspot;

vec2 scale   = EMU_DEFAULT ( vec2, Emu_Quantise_RGBA_scale,   vec2(1.0) );
vec2 hotspot = EMU_DEFAULT ( vec2, Emu_Quantise_RGBA_hotspot, vec2(0.0) );

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// The colour quantisation parameters. Each is a four component vector which
// allows different values to be used for each of the red, green,  blue  and
// alpha colour channels. Quantisation is only performed for a channel if QN
// for thst chsnnel is greater that 0.0.  It is expected that the same value
// of QN would normaly be used for each of the R, G and B channels  and  0.0
// used for the alpha channel so that it is not affected by the shader.

uniform vec4 Emu_Quantise_RGBA_QN; // Quantisation factor
uniform vec4 Emu_Quantise_RGBA_Q0; // Quantisation offset

#define QN Emu_Quantise_RGBA_QN
#define Q0 Emu_Quantise_RGBA_Q0

// ============================================================================
// == The shader's major function =============================================
// ============================================================================

vec4 Emu_Quantise_RGBA ( vec2 uv )
 {
   vec4 c = texture2D ( iChannel0, uv );

   if ( QN.r > 0.0 ) c.r = floor ( c.r * QN.r + Q0.r ) / QN.r;
   if ( QN.g > 0.0 ) c.g = floor ( c.g * QN.g + Q0.g ) / QN.g;
   if ( QN.b > 0.0 ) c.b = floor ( c.b * QN.b + Q0.b ) / QN.b;
   if ( QN.a > 0.0 ) c.a = floor ( c.a * QN.a + Q0.a ) / QN.a;

   return c;

 }

// ============================================================================
// == The shader's main routine ===============================================
// ============================================================================

void main ( void )
 {
   // Get the normalised coordinates of the current point.

   vec2 uv = Emu_Normalise_to_Window ( gl_FragCoord.xy );

   uv = ( uv - hotspot ) / scale + hotspot;

   // Perform the mixing and update the shader's outputs.

   gl_FragColor = Emu_Quantise_RGBA(uv) * gl_Color;

}